// $Id: CThreadManager.cpp,v 1.6 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CThreadManager.hpp"
#include "../Exceptions/CException.hpp"
#include "../Testing/CTrace.hpp"
using Exponent::Exceptions::CException;
using Exponent::Threading::CThreadManager;
using Exponent::Testing::CTrace;

#ifndef WIN32
#include <unistd.h>
#include <pthread.h>
#endif

//	===========================================================================
void CThreadManager::startThread(IThread *thread)
{
	CTrace::trace("Starting Thread");

	// Check the thread is valid
	if (thread && !thread->isThreadActive())
	{
		
		CTrace::trace("Getting Thread Handle");
	
		// Create the thread handle
		IThread::SThreadHandle *threadHandle = thread->getThreadHandle();

		// If its null, then give them one
		if (threadHandle == NULL)
		{
			// Create a new one
			threadHandle = new IThread::SThreadHandle;
			
			// NULL the thread pointer
			NULL_POINTER(threadHandle->m_threadHandle);

			// Store the handle
			thread->setThreadHandle(threadHandle);
		}
	#ifdef WIN32
		// Create the thread giving it our function and the thread as a pointer
		threadHandle->m_threadHandle = CreateThread(NULL, 0, threadCallback, (void *)thread, 0, NULL);
	#else
		CTrace::trace("Creating Thread Handle");
		// Create the thread handle
		threadHandle->m_threadHandle = new pthread_t;
		// Create the thread giving it our function and the thread as a pointer
		if (pthread_create(threadHandle->m_threadHandle, NULL, threadCallback, (void *)thread) != 0)
		{
			CTrace::trace("FAiled in creating Thread Handle");
			throw CException("Failed to create exception!", "CThreadManager::startThread(IThread*)");
		}
	#endif
	}
}

//	===========================================================================
void CThreadManager::stopThread(IThread *thread)
{
	CTrace::trace("A request has been made to stop the thread = 0x%p", thread);

	// Check the thread is valid
	if (thread)
	{
		// Store the handle to the thread
		IThread::SThreadHandle *threadHandle = thread->getThreadHandle();

		// If the thread is active we want to stop it
		if (thread->isThreadActive())
		{
			CTrace::trace("The thread is active, closing");

			// Forcibly close the thread
			thread->stopThread();

			// If we got the handle
			if (threadHandle && threadHandle->m_threadHandle)
			{
			#ifdef WIN32
				// TerminateThread(threadHandle->m_threadHandle, 0);
			#else
			
			#endif
			}
		}

		// Null the handle
		if (threadHandle)
		{
			// NULL the thread handle
			threadHandle->m_threadHandle = NULL;
		}
	}
}

//	===========================================================================
void CThreadManager::terminateThread(IThread *thread)
{
		// Check the thread is valid
	if (thread)
	{
		// Store the handle to the thread
		IThread::SThreadHandle *threadHandle = thread->getThreadHandle();

		// If the thread is active we want to stop it
		if (thread->isThreadActive())
		{
			// Forcibly close the thread
			thread->stopThread();
			if (threadHandle && threadHandle->m_threadHandle)
			{
			#ifdef WIN32
				TerminateThread(threadHandle->m_threadHandle, 0);
			#else
			
			#endif
			}
		}

		// Null the handle
		if (threadHandle)
		{
			// NULL the thread handle
			threadHandle->m_threadHandle = NULL;
		}
	}
}

//	===========================================================================
#ifdef WIN32
unsigned long __stdcall CThreadManager::threadCallback(void *data)
#else
void *CThreadManager::threadCallback(void *data)
#endif
{
	CTrace::trace("In thread callback function");

	// Check that the data is valid
	if (data)
	{
		// STore the thread
		IThread *thread = (IThread *)data;
		CTrace::trace("Acquired thread handle from the input data pointer");

		// Store the handle to the thread
		IThread::SThreadHandle *threadHandle = thread->getThreadHandle();

		// We failed!!
		if (threadHandle == NULL || threadHandle->m_threadHandle == NULL)
		{
			CTrace::trace("Thread is null, cannot continue");
			return 0;
		}

		// Store the notification unit
		IThreadListener *listener = thread->getThreadListener();

		// Assume failure
		bool sucessfull = false;	

		try
		{
			CTrace::trace("Trying thread run");

			//  Do the thread process
			sucessfull = thread->runThread();
		}
		catch(CException myException)
		{
			CTrace::trace(myException.getErrorReason(), "Exception during thread run ");
			CTrace::trace(myException.getFunctionName(), "In function ");

			// Notify the listener is available
			if (listener)
			{
				listener->threadHasAborted(thread);
			}
			
			// Throw on the exception
			throw myException;
		}
		catch(...)
		{
			CTrace::trace("Unknown Exception during thread run");

			// Notify the listener is available
			if (listener)
			{
				listener->threadHasAborted(thread);
			}

			// Throw a proper exponent exception
			throw CException("Caught unkown exception", "CThreadManager::threadCallback(void*)");
		}

		// If its valid notify it that the process is just about to start
		if (listener)
		{
			if (sucessfull)
			{
				// Notify the listener that we are complete..
				listener->threadHasCompleted(thread);
			}
			else
			{
				// Notify that there was a problem
				listener->threadHasAborted(thread);
			}
		}

		CTrace::trace("Stopping the thread");

		// Stop the thread, its all done now :)
		thread->stopThread();

		CTrace::trace("NULLing the pointer");

		// NULL the pointer so that thread postion cant be updated. This is release by the host on return
		NULL_POINTER(threadHandle->m_threadHandle);
	}

	CTrace::trace("Leaving Thread Process");
	return 0;
}